<?php
use Livewire\Volt\Component;
use App\Models\FastEvent;
use Carbon\Carbon;

new class extends Component {
    public $activeEvents = [];

    /**
     * تهيئة البيانات عند تحميل الكومبوننت
     */
    public function mount()
    {
        $this->loadActiveEvents();
    }

    /**
     * تحميل الأحداث النشطة فقط
     */
    public function loadActiveEvents()
    {
        try {
            // جلب الأحداث النشطة فقط والتي لم تنته صلاحيتها
            $events = FastEvent::where('is_active', true)
                ->where(function($query) {
                    $query->whereNull('expired_at')
                          ->orWhere('expired_at', '>', Carbon::now());
                })
                ->orderBy('created_at', 'desc')
                ->get();

            $this->activeEvents = $events->toArray();
        } catch (\Exception $e) {
            $this->activeEvents = [];
            session()->flash('error', 'Error loading events: ' . $e->getMessage());
        }
    }

    /**
     * تحديث البيانات
     */
    public function refreshEvents()
    {
        $this->loadActiveEvents();
        session()->flash('success', 'Events refreshed successfully!');
    }

    /**
     * الحصول على محتوى blade للحدث
     */
    public function getEventContent($bladeName)
    {
        if (!$bladeName) return null;

        $bladePath = resource_path("views/dashboard/fastEvents/{$bladeName}.blade.php");

        if (file_exists($bladePath)) {
            try {
                return view("dashboard.fastEvents.{$bladeName}")->render();
            } catch (\Exception $e) {
                return null;
            }
        }

        return null;
    }

    /**
     * API للحصول على الأحداث النشطة
     */
    public function getActiveEventsData()
    {
        return response()->json([
            'events' => $this->activeEvents,
            'count' => count($this->activeEvents)
        ]);
    }
};
?>
<div>

    <div x-data="eventManager" class="fast-events-container">
        {{-- Loading Overlay --}}
        <div x-show="loading" class="loading-overlay">
            <div class="loading-spinner">
                <div class="spinner-ring"></div>
                <p data-translate="fastEvents.loading">Loading events...</p>
            </div>
        </div>

        {{-- Main Header --}}
        <header class="fast-events-header">
            <div class="header-content">
                <div class="header-icon">
                    <i class="fas fa-calendar-star"></i>
                </div>
                <div class="header-text">
                    <h1 class="header-title" data-translate="fastEvents.title">Active Events</h1>
                    <p class="header-subtitle" data-translate="fastEvents.subtitle">Live Gaming Events & Rewards</p>
                </div>
                <button
                    @click="refreshEvents"
                    class="refresh-btn"
                    :class="{ 'loading': refreshing }"
                    :disabled="refreshing"
                >
                    <i class="fas" :class="refreshing ? 'fa-spinner fa-spin' : 'fa-sync-alt'"></i>
                    <span data-translate="fastEvents.refreshing" x-show="refreshing">Refreshing...</span>
                    <span data-translate="fastEvents.refresh" x-show="!refreshing">Refresh Events</span>
                </button>
            </div>
        </header>

        {{-- Success Messages --}}
        @if (session()->has('success'))
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <span>{{ session('success') }}</span>
            </div>
        @endif

        {{-- Error Alert --}}
        @if (session()->has('error'))
            <div class="alert alert-error">
                <i class="fas fa-exclamation-triangle"></i>
                <span>{{ session('error') }}</span>
            </div>
        @endif

        {{-- Events Grid --}}
        <div x-show="!loading && events.length > 0" class="events-grid">
            @if($activeEvents && count($activeEvents) > 0)
                @foreach($activeEvents as $index => $event)
                    <div class="event-card" @click="openEventModal({{ json_encode($event) }})">
                        <div class="card-glow"></div>

                        {{-- Card Header --}}
                        <div class="event-card-header">
                            <div class="event-status">
                                <div class="status-dot active"></div>
                                <span data-translate="fastEvents.eventCard.active">Active</span>
                            </div>
                            @if($event['expired_at'])
                                <div class="event-timer">
                                    <i class="fas fa-clock"></i>
                                    <span>{{ \Carbon\Carbon::parse($event['expired_at'])->diffForHumans() }}</span>
                                </div>
                            @endif
                        </div>

                        {{-- Card Body --}}
                        <div class="event-card-body">
                            <div class="event-icon">
                                <i class="fas fa-star"></i>
                            </div>
                            <h3 class="event-title">{{ $event['event_name'] }}</h3>
                            @if($event['blade_name'])
                                <div class="event-type">
                                    <i class="fas fa-bolt"></i>
                                    <span>{{ ucfirst(str_replace('-', ' ', $event['blade_name'])) }}</span>
                                </div>
                            @else
                                <div class="event-type">
                                    <i class="fas fa-gift"></i>
                                    <span>Fast Event</span>
                                </div>
                            @endif
                        </div>

                        {{-- Card Footer --}}
                        <div class="event-card-footer">
                            <button class="view-event-btn" data-translate="fastEvents.eventCard.viewEvent">
                                View Event
                            </button>
                        </div>
                    </div>
                @endforeach
            @endif
        </div>

        {{-- No Events State --}}
        @if(!$activeEvents || count($activeEvents) == 0)
            <div class="no-events-state">
                <div class="no-events-icon">
                    <i class="fas fa-calendar-times"></i>
                </div>
                <h2 class="no-events-title" data-translate="fastEvents.noEvents">No Active Events</h2>
                <p class="no-events-description" data-translate="fastEvents.noEventsMessage">
                    There are currently no active events. Check back later for new adventures!
                </p>
            </div>
        @endif

        {{-- Event Modal --}}
        <div x-show="selectedEvent" class="event-modal-overlay" @click.self="closeEventModal">
            <div class="event-modal">
                {{-- Modal Header --}}
                <div class="modal-header">
                    <h2 class="modal-title" x-text="selectedEvent?.event_name" data-translate="fastEvents.eventModal.eventDetails">Event Details</h2>
                    <button @click="closeEventModal" class="modal-close-btn" data-translate="fastEvents.eventModal.close">
                        <i class="fas fa-times"></i>
                    </button>
                </div>

                {{-- Modal Body --}}
                <div class="modal-body">
                    <div class="event-content">
                        {{-- Loading state --}}
                        <div x-show="loadingEventContent" class="blade-content-placeholder">
                            <div class="loading-blade">
                                <div class="spinner-ring"></div>
                                <p data-translate="fastEvents.eventModal.loadingContent">Loading event content...</p>
                            </div>
                        </div>

                        {{-- Event content will be loaded here dynamically --}}
                        <div x-show="!loadingEventContent && eventContent" x-html="eventContent"></div>

                        {{-- No content message --}}
                        <div x-show="!loadingEventContent && !eventContent" class="no-content-message">
                            <i class="fas fa-info-circle"></i>
                            <p data-translate="fastEvents.eventModal.noContent">
                                Event content is not available at the moment. Please try again later.
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('alpine:init', () => {
            Alpine.data('eventManager', () => ({
                events: @json($activeEvents),
                loading: false,
                refreshing: false,
                error: '',
                selectedEvent: null,
                eventContent: '',
                loadingEventContent: false,

                init() {
                    // التطبيق جاهز للاستخدام
                },

                async refreshEvents() {
                    this.refreshing = true;
                    try {
                        // استدعاء Livewire method
                        await @this.call('refreshEvents');
                        this.events = @this.get('activeEvents');
                    } catch (error) {
                        console.error('Error refreshing events:', error);
                    } finally {
                        this.refreshing = false;
                    }
                },

                async openEventModal(event) {
                    this.selectedEvent = event;
                    this.eventContent = '';
                    this.loadingEventContent = true;

                    // إذا كان للحدث blade_name، نحمل المحتوى
                    if (event.blade_name) {
                        try {
                            const content = await @this.call('getEventContent', event.blade_name);
                            this.eventContent = content;
                        } catch (error) {
                            console.error('Error loading event content:', error);
                            this.eventContent = null;
                        }
                    }

                    this.loadingEventContent = false;
                },

                closeEventModal() {
                    this.selectedEvent = null;
                    this.eventContent = '';
                },

                getTimeRemaining(expiredAt) {
                    if (!expiredAt) return 'No expiry';

                    const now = new Date();
                    const expiry = new Date(expiredAt);
                    const diff = expiry - now;

                    if (diff <= 0) return 'Expired';

                    const days = Math.floor(diff / (1000 * 60 * 60 * 24));
                    const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
                    const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));

                    if (days > 0) return `${days}d ${hours}h`;
                    if (hours > 0) return `${hours}h ${minutes}m`;
                    return `${minutes}m`;
                }
            }));
        });
    </script>
</div>
